/***************************************************************
 *                   Materials Object Library                  *
 * Class transversely_isotropic_tensor: declaration for        *
 *          transversely isotropic (in x3 direction) materials *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.0.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2008,2009 BERBENNI Stéphane
 * copyright © 2009,2011,2012 COLLARD Christophe
 * copyright © 2008,2009,2011,2012 Centre National de la Recherche Scientifique
 * copyright © 2008,2009,2011 Arts et Métiers ParisTech
 * copyright © 2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2011 Laboratoire d'Etude des Microstructures et de Mécanique des Matériaux (LEM3 - CNRS)
 * copyright © 2011,2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::transversely_isotropic_tensor
    \brief transversely isotropic tensor library \n

    \htmlonly 
    <FONT color="#838383">

    transversely isotropic tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    We define a transversely isotropic tensor \f$ T \f$ in \f$ x_3 \f$ direction through five constants: \f$ c_{11}, c_{33}, c_{66}, c_{12}, c_{13} \f$. Its non zero components read \n
    \f$ \displaystyle T_{1111} = T_{2222} = c_{11} \f$ \n
    \f$ T_{3333} = c_{33} \f$ \n
    \f$ \displaystyle T_{1212} = T_{1221} = T_{2112} = T_{2121} = \frac{1}{2} \left( c_{11} - c_{12} \right) \f$ \n
    \f$ T_{1313} = T_{1331} = T_{3113} = T_{3131} = T_{2323} = T_{2332} = T_{3223} = T_{3232} = c_{66} \f$ \n
    \f$ \displaystyle T_{1122} = T_{2211} = c_{12} \f$ \n
    \f$ \displaystyle T_{1133} = T_{3311} = T_{2233} = T_{3322} = c_{13} \f$ \n

    The associated reduced symmetric matrix reads (see detailed description of transversely_isotropic_elastic_tensor) : \n
     \f$  \begin{pmatrix} c_{11} & c_{12} & c_{13} & 0      & 0      & 0 \\ 
                                 & c_{11} & c_{13} & 0      & 0      & 0 \\
			         &        & c_{33} & 0      & 0      & 0 \\
			         & sym    &        & \frac{1}{2} (c_{11} - c_{12}) & 0      & 0 \\
			         &        &        &        & c_{66} & 0 \\
			         &        &        &        &        & c_{66} \\
        \end{pmatrix} 
	\f$ \n
	and we get the following symmetries : \f$T_{ijkl} = T_{jikl} = T_{ijlk} = T_{klij} \f$.

    \author copyright \htmlonly &#169; \endhtmlonly 2008, 2009 Stephane BERBENNI \n
            copyright \htmlonly &#169; \endhtmlonly 2009, 2011, 2012 Christophe COLLARD \n
	    copyright \htmlonly &#169; 2009, 2010, 2011, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2009, 2010, 2011 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2008, 2009, 2011 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2011 Laboratoire d'Etude des Microstructures et de M&#233;canique des Mat&#233;riaux (LEM3 - CNRS) \endhtmlonly \n
            copyright \htmlonly &#169; 2011, 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.0.0
    \date 2008-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type transversely isotropic tensor
#endif

#ifndef __transversely_isotropic_tensors_hpp
#define __transversely_isotropic_tensors_hpp


#ifndef __stdlib_h
#include <stdlib.h>
#endif

#ifndef __iostream
#include <iostream>
#endif

#ifndef __fstream
#include <fstream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __math_h
#include <math.h>
#endif

#ifndef __string_h
#include <string.h>
#endif

#ifndef __maths_hpp
#include "MOL++/maths.hpp"
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifdef __symtensors4_hpp
#include "MOL++/symtensors4.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//===========================================================================
template <class T> class transversely_isotropic_tensor : public symtensor4<T>
//===========================================================================
{
  using symtensor4<T>::mat;

  protected :
    T C11, C33, C66, C12, C13; // material constants : 5 with C66=(C11-C12)/2
    string tensor_name;

    void build_tensor ();  // computes the 4th order isotropic transverse tensor

  public :
    transversely_isotropic_tensor (string = "");  // default constructor
    transversely_isotropic_tensor (T, T, T, T, T, string = "");
    transversely_isotropic_tensor (const vector<T>&, string = "");
    transversely_isotropic_tensor (const tensor4<T>&, string = "");  // cast conversion
    ~transversely_isotropic_tensor () {}  // destructor

    string name(string = "");  // returns / computes tensor's name
    vector<T> elastic_constants ();
    transversely_isotropic_tensor<T> inv ();
    tensor4<T> change_basis (T, T, T);

    template <class Tf> friend ostream& operator << (ostream&, transversely_isotropic_tensor<Tf>&);
};


//==== Private methods for transversely isotropic tensors ===================================


//-----------------------------------------------------------------------
template <class T> void transversely_isotropic_tensor<T>::build_tensor ()
//-----------------------------------------------------------------------
{
  mat(1,1) = mat(2,2) = C11;
  mat(3,3) = C33;
  mat(4,4) = 0.5 * (C11 - C12);
  mat(5,5) = mat(6,6) = C66;
  mat(1,2) = mat(2,1) = C12;
  mat(1,3) = mat(3,1) = mat(2,3) = mat(3,2) = C13;
}


//=====Public methods for cubic tensor==================================================


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order transversely isotropic tensors.

  \param name tensor name
*/

//-----------------------------------------------------------------------------------------------------------------
template <class T> transversely_isotropic_tensor<T>::transversely_isotropic_tensor (string name) : symtensor4<T>(3)
//-----------------------------------------------------------------------------------------------------------------
{
  C11 = C33 = C66 = C12 = C13 = 0;  // 5 coeff
  tensor_name = name;
}


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order transversely isotropic tensors.

  (see detailed description).

  \param c11 \f$ c_{11} \f$ value
  \param c33 \f$ c_{33} \f$ value
  \param c66 \f$ c_{66} \f$ value
  \param c12 \f$ c_{12} \f$ value
  \param c13 \f$ c_{13} \f$ value
  \param name tensor name
*/

//---------------------------------------------------------------------------------------------------------------------------------
template <class T>
transversely_isotropic_tensor<T>::transversely_isotropic_tensor (T c11, T c33, T c66, T c12, T c13, string name) : symtensor4<T>(3)
//---------------------------------------------------------------------------------------------------------------------------------
{
  C11 = c11;
  C33 = c33;
  C66 = c66;
  C12 = c12;
  C13 = c13;
  build_tensor();
  tensor_name = name;
}


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order transversely isotropic tensors.

  Computes \f$ C_{ijkl} \f$ (see detailed description).


  \param v \f$ (c_{11} , c_{33} , c_{66} , c_{12} , c_{13} ) \f$ values
  \param name tensor name
*/

//-------------------------------------------------------------------------------------------------------------------------------------
template <class T> transversely_isotropic_tensor<T>::transversely_isotropic_tensor (const vector<T>& v, string name) : symtensor4<T>(3)
//-------------------------------------------------------------------------------------------------------------------------------------
{
  assert (v.dim() == 5);

  C11 = v[1];
  C33 = v[2];
  C66 = v[3];
  C12 = v[4];
  C13 = v[5];
  
  build_tensor();
  tensor_name = name;
}


/*!
  \brief Constructor for cast conversion.

  Converts a \f$ \displaystyle 4^{th} \f$ order tensor into a transversely isotropic tensor.

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param name tensor name
*/

//-------------------------------------------------------------------------------------------------------------------
template <class T> transversely_isotropic_tensor<T>::transversely_isotropic_tensor (const tensor4<T>& t, string name)
//-------------------------------------------------------------------------------------------------------------------
{
  assert(mat.nbRows() && mat.nbColumns());

  C11 = mat(1,1);
  C33 = mat(3,3);
  C66 = mat(6,6);
  C12 = mat(1,2);
  C13 = mat(1,3);
  tensor_name = name;
  build_tensor();

  assert ((tensor4<T>)(*this) == t);
}



/*!
  \brief Changes and/or returns tensor name.
  \param name tensor name
  \return tensor name
*/

//----------------------------------------------------------------------------
template <class T> string transversely_isotropic_tensor<T>::name (string name)
//----------------------------------------------------------------------------
{
  if (name != "") tensor_name = name;

  return tensor_name;
}


/*!
  \brief Returns elastic constants.
  \return \f$ (c_{11}, c_{33}, c_{66}, c_{12}, c_{13}) \f$
*/

//---------------------------------------------------------------------------------
template <class T> vector<T> transversely_isotropic_tensor<T>::elastic_constants ()
//---------------------------------------------------------------------------------
{
  vector<T> constants (5,false);
  constants[1] = C11;
  constants[2] = C33;
  constants[3] = C66;
  constants[4] = C12;
  constants[5] = C13;

  return constants;
}


/*!
  \brief Computes the inverse of a \f$ \displaystyle 4^{th} \f$ order transversely isotropic tensor.

  \return inverse of the tensor which is also a transversely isotropic tensor
*/

//------------------------------------------------------------------------------------------
template <class T> transversely_isotropic_tensor<T> transversely_isotropic_tensor<T>::inv ()
//------------------------------------------------------------------------------------------
{
  T det_inv = 1 / (C33 * (pow (C11, 2) - pow (C12, 2))  +  2 * pow (C13, 2) * (C12 - C11));
  T C_11 = (C11 * C33  -  pow (C13, 2)) * det_inv;
  T C_33 = (pow (C11, 2) - pow (C12, 2)) * det_inv;
  T C_12 = (pow (C13, 2) - C12 * C33) * det_inv;
  T C_13 = (C12 * C13 - C11 * C13) *det_inv;

  return transversely_isotropic_tensor<T> (C_11, C_33, 0.25/C66, C_12, C_13); // l'encodage (symtensor -> matrix) utilise n'est pas celui de la doc => laisser le 0.25 / C66 !!
}


/*!
  \brief Computes the change of coordinates of a \f$ \displaystyle 4^{th} \f$ order transversely isotropic tensor.

  \param phi1 Euler angle \f$ \varphi_1 \f$
  \param Phi Euler angle \f$ \Phi \f$
  \param phi2 Euler angle \f$ \varphi_2 \f$

  \return C in the new basis
*/

//--------------------------------------------------------------------------------------------------
template <class T> tensor4<T> transversely_isotropic_tensor<T>::change_basis (T phi1, T Phi, T phi2)
//--------------------------------------------------------------------------------------------------
{
  symmatrix<T> C(6,6);
  //  Phi /= 180 * pi();
  T s2 = pow (sin (Phi), 2);
  T s4 = pow (s2, 2);
  T coef = sqrt((T) 2);

  C(1,1) = C11;
  C(1,2) = C12 + (C13 - C12) * s2;
  C(1,3) = C13 + (C12 - C13) * s2;
  C(2,2) = C11 + s2 * (2 * (C13 + 2 * C66) * (1 - s2) + (C33 + C11) * s2 - 2*C11);
  C(2,3) = .125 * (C11 + 6*C13 - 4*C66 + C33 - (C11 - 2*C13 - 4*C66 + C33) * cos(4*Phi));
  C(3,3) = C33 + s2 * (2 * (C13 + 2 * C66) * (1 - s2) + (C33 + C11) * s2 - 2*C33);
  C(1,6) = (C12 - C13) * cos(Phi) * sin(Phi) * coef;
  C(2,6) = .25 * coef * (C11 - C33 + (C11 - 2*C13 - 4*C66 + C33) * cos(2*Phi)) * sin(2*Phi);
  C(3,6) = .25 * coef * (C11 - C33 - (C11 - 2*C13 - 4*C66 + C33) * cos(2*Phi)) * sin(2*Phi);
  C(6,6) = .25 * (C11 - 2*C13 + 4*C66 + C33 - (C11 - 2*C13 - 4*C66 + C33) * cos(4*Phi));
  C(5,5) = 2 * (C66 + (.5 * (C11 - C12) - C66) * s2);
  C(5,4) = .5 * (C11 - C12 - 2* C66) * sin (2*Phi);
  C(4,4) = (C11 - C12) + (2*C66 + C12 - C11) * s2;

  return matrix2symtensor(C);
}


//-------------------------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, transversely_isotropic_tensor<Tf>& tsr)
//-------------------------------------------------------------------------------------------
{
  assert (tsr.C11 * tsr.C66 * tsr.C12);

  s << tsr.tensor_name << endl;
  s << "Material constants : (" << tsr.C11 << ", " << tsr.C33 << ", " << tsr.C66 << ", " << tsr.C12 << ", " << tsr.C13 << ")" << endl;
  s << (tensor4<Tf>) tsr;
  s << endl;

  return s;
}


}


#endif
